(*************************************************************************
 *  UndoU.pas                                                            *
 *  Vladimr Slvik 2005-10                                              *
 *  Delphi 7 Personal                                                    *
 *  cp1250                                                               *
 *                                                                       *
 *  undo structures for Shades :                                         *
 *    ?                                                                  *
 *                                                                       *
 *  -additional libraries: Graphics32                                    *
 *************************************************************************)

unit UndoU;

{$INCLUDE ..\Switches.inc}
{t default -}

//------------------------------------------------------------------------------

interface

uses GR32, Types, Classes,
     ClassBaseU;

//------------------------------------------------------------------------------

type TUndoItem = class
     private
       FOld, FNew: TBitmap32;
       FArea: TRect;
       FEngine: TPictureEngineBase;
     public
       constructor Create(const AEngine: TPictureEngineBase; const AWhere: TRect);
       destructor Destroy; override;
       procedure ApplyForward;  // redo
       procedure ApplyBackward; // undo
     published
       property Area: TRect read FArea;
     end;

//------------------------------------------------------------------------------

type TUndoEngine = class(TUndoEngineBase)
       constructor Create(AEngine: TPictureEngineBase);
       // every instance is bound to its owner engine once instantiated
       destructor Destroy; override;
     private
       FUndos: TList;
       FEngine: TPictureEngineBase;
       FInternalActionActive: Boolean; // flag for preventing "false paralelism"
     protected
       function GetCount: Integer; override;
       function GetAtEnd: Boolean; override;
       function GetAtStart: Boolean; override;
       procedure Crop; override;
       procedure SetRestrictCount(Value:Boolean); override;
       procedure SetMaxCount(Value: Integer); override;
     public
       procedure AddStep(const ChangeRect: TRect); override;
       procedure Truncate(const Index: Integer); override;
       procedure Restart; override;
       procedure Undo; override;
       procedure Redo; override;
     published
       property Count;
       property AtEnd;
       property AtStart;
       property Position;
       property MaxCount;
       property RestrictCount;
     end;

//==============================================================================
implementation

uses SysUtils, Contnrs, Math, TypInfo,
     CoreTypeU, CoreLowU, CoreEngineU, CalcUtilU,
     TranslationU;

//------------------------------------------------------------------------------

procedure TUndoEngine.Redo;
var U: TUndoItem;
begin
  if FUndos.Count < 1 then
    raise EAbort.Create(_('No redo possible - no steps')); //t+
  if FPosition >= FUndos.Count then
    raise EAbort.Create(_('No redo possible - at end')); //t+
  // make sure redo is allowed operation
  U:= FUndos.Items[FPosition];
  U.ApplyForward;
  Inc(FPosition);
end;

//------------------------------------------------------------------------------

procedure TUndoEngine.Undo;
var U: TUndoItem;
begin
  if FUndos.Count < 1 then
    raise EAbort.Create(_('No undo possible - no steps')); //t+
  if FPosition < 1 then
    raise EAbort.Create(_('No undo possible - at step #0')); //t+
  // make sure undo is allowed operation
  Dec(FPosition);
  U:= FUndos.Items[FPosition];
  U.ApplyBackward;
end;

//------------------------------------------------------------------------------

procedure TUndoEngine.Crop;
// ensure the undo item list does not exceed maximal allowed size
begin
  with FUndos do if FRestrictCount then while Count > FMaxCount do begin
    FUndos.Delete(0);
    Dec(FPosition);
  end;
  if FPosition < 0 then FPosition:= 0;
end;

//------------------------------------------------------------------------------

procedure TUndoEngine.SetMaxCount(Value: Integer);
// setter for max. count
begin
  if Value < 1 then
    raise Exception.CreateFmt('Value %d is lower than 1 in TUndoEngine.SetMaxCount.', [Value]);
  FMaxCount:= Value;
  Crop;
end;

//------------------------------------------------------------------------------

procedure TUndoEngine.SetRestrictCount(Value: Boolean);
// setter for max. count enable
begin
  FRestrictCount:= Value;
  Crop;
end;

//------------------------------------------------------------------------------

procedure TUndoEngine.Restart;
// free all and start freshly anew
begin
  FUndos.Clear;
  FPosition:= 0;
end;

//------------------------------------------------------------------------------

function TUndoEngine.GetAtStart: Boolean;
// getter for "head is at first item"
begin
  Result:= FPosition = 0;
end;

//------------------------------------------------------------------------------

function TUndoEngine.GetAtEnd: Boolean;
// getter for "head is behind last item"
begin
  Result:= FUndos.Count = FPosition;
end;

//------------------------------------------------------------------------------

function TUndoEngine.GetCount: Integer;
begin
  Result:= FUndos.Count;
end;

//------------------------------------------------------------------------------

procedure TUndoEngine.Truncate(const Index: Integer);
// cut everything behind given position
begin
  // index is the item to be filled in next step
  // or current position of "head" in engine
  // if index = 0, all is deleted
  if not InRange(Index, 0, FUndos.Count - 1) then
    raise Exception.CreateFmt('Index %d out of bounds in TUndoEngine.Truncate!', [Index]);
    // crash on bad index
  with FUndos do while Count > Index do Delete(Count - 1);
  // Go backward from end and delete until head stands on index = after index - 1
  // and count is equal -> head also stands after the last item.
  // FUndos is TObjectList and frees its items automatically!
end;

//------------------------------------------------------------------------------

procedure TUndoEngine.AddStep(const ChangeRect: TRect);
// add new undo step
var U: TUndoItem;
    NewBounds: TRect;
begin
  if IsRectEmpty(ChangeRect) then Abort; // die on nothing to do
  if not IntersectRect(NewBounds, ChangeRect, FEngine.BoundsRect) then Abort;
  // constrain working area to picture, die on nothing to do
  if not GetAtEnd then Truncate(FPosition); // kill everything after current position
  U:= TUndoItem.Create(FEngine, NewBounds); // new item
  FUndos.Add(U); // add new item to list
  Inc(FPosition);
  Crop; // obey count limit
end;

//------------------------------------------------------------------------------

destructor TUndoEngine.Destroy;
begin
  FreeAndNil(FUndos);
  inherited Destroy;
end;

//------------------------------------------------------------------------------

constructor TUndoEngine.Create(AEngine: TPictureEngineBase);
begin
  if not Assigned(AEngine) then
    raise Exception.Create('No TPictureEngineBase instance passed to TUndoEngine.Create!');
  inherited Create;
  FUndos:= TObjectList.Create;
  FEngine:= AEngine;
  FPosition:= 0;
  FRestrictCount:= False;
  FMaxCount:= 10; // hardcoded
  FInternalActionActive:= False;
end;

//==============================================================================

constructor TUndoItem.Create(const AEngine: TPictureEngineBase; const AWhere: TRect);
var dx, dy: Integer;
begin
  inherited Create;
  FEngine:= AEngine;
  FArea:= AWhere;
  FOld:= TBitmap32.Create;
  FNew:= TBitmap32.Create;
  FOld.DrawMode:= dmOpaque;
  FNew.DrawMode:= dmOpaque;
  dx:= RectWidth(FArea);
  dy:= RectHeight(FArea);
  FOld.SetSize(dx, dy);
  FNew.SetSize(dx, dy);
  FEngine.Picture.DrawMode:= dmOpaque; // because it needs to overwrite even with transparency
  FEngine.Overlay.DrawMode:= dmOpaque;
  FOld.Draw(0, 0, FArea, FEngine.Picture);
  FNew.Draw(0, 0, FArea, FEngine.Overlay);
end;

//------------------------------------------------------------------------------

destructor TUndoItem.Destroy;
begin
  FOld.Free;
  FNew.Free;
  inherited Destroy;
end;

//------------------------------------------------------------------------------

procedure TUndoItem.ApplyForward; // as REDO
begin
  // when going forward, dmBlend since Overlay is normally applied that way
  FNew.DrawMode:= dmBlend;
  with FEngine do begin
    FNew.DrawTo(Picture, FArea);
    MakeTransparent(Picture, BackClrOpaque, BoundsRect);
    with FArea do Overlay.FillRect(Left, Top, Right, Bottom, BackClrTransparent);
    // clean overlay - modifying Picture can succed but be hidden by leftovers
    // in there!
    UpdateRect(FArea);
    RedrawRect(FArea);
  end;
end;

//------------------------------------------------------------------------------

procedure TUndoItem.ApplyBackward; // as UNDO
begin
  // when going backward, dmOpaque since the old part was replaced and must be
  // re-replaced to look correctly again
  FOld.DrawMode:= dmOpaque;
  with FEngine do begin
    FOld.DrawTo(Picture, FArea);
    MakeTransparent(Picture, BackClrOpaque, BoundsRect);
    with FArea do Overlay.FillRect(Left, Top, Right, Bottom, BackClrTransparent);
    UpdateRect(FArea);
    RedrawRect(FArea);
  end;
end;

//------------------------------------------------------------------------------

end.
