(*************************************************************************
 *  CoreEngineU.pas                                                      *
 *  Vladimr Slvik 2007-10                                              *
 *  Delphi 7 Personal                                                    *
 *  cp1250                                                               *
 *                                                                       *
 *  The top-level component of image machinery. Groups more or less      *
 *    loosely other parts like background "provider", undo and such.     *
 *                                                                       *
 *  -additional libraries: Graphics32                                    *
 *************************************************************************)

unit CoreEngineU;

{$INCLUDE ..\Switches.inc}
{t default -}

//------------------------------------------------------------------------------

interface

uses Types, Classes,
     GR32, GR32_Image,
     CoreTypeU, ClassBaseU, ToolOptionsU, UndoU, BackgroundEngineU, ConfigU;
//------------------------------------------------------------------------------

type TPictureEngine = class(TPictureEngineBase)
       constructor Create;
       destructor Destroy; override;
       procedure AfterConstruction; override;
     private
       FBackgroundEngine: TBackgroundEngine;
       FPctBuffer: TBitmap32;
       // top-level cache, depends on mode
       FViewMode: TViewMode;
       // day, night, special
       FDoCallBack: Boolean;
       // respond to callbacks (now from background engine) or leave them be?
       procedure RepaintBuffer(const APlace: TRect);
       procedure ResyncOverlay;
     protected
       procedure SetViewport(ANew: TImgView32); override;
       procedure SetViewMode(AMode: TViewMode);
       function GetBoundsRect: TRect; override;
       procedure SetOptions(const AOptions: TSdiEdOptions); override;
     public
       property Picture;
       property Overlay;
       property Viewport;
       property BoundsRect; 
       property Options;
       property ToolOptions;
       property UndoEngine;
       property OnChange;
       property Background: TBackgroundEngine read FBackgroundEngine;
       property ViewMode: TViewMode read FViewMode write SetViewMode;
       procedure BackgroundChangeHandler(Sender: TObject);
       procedure Reloaded; override;
       procedure Paint; override;
       procedure RedrawRect(const APlace: TRect); override;
       procedure UpdateRect(const APlace: TRect); override;
       procedure CommitOverlay(const AWhere: array of TRect); override;
       procedure DiscardOverlay; override;
       procedure CopyOverlay; override;
       procedure Dump; override;
     end;

//==============================================================================
implementation

uses SysUtils, Math, CalcUtilU, CoreLowU;
//------------------------------------------------------------------------------

constructor TPictureEngine.Create;
begin
  inherited Create;
  FDoCallBack:= True;
  FPctOriginal:= TBitmap32.Create;
  FPctOverlay:= TBitmap32.Create;
  FPctBuffer:= TBitmap32.Create;
  FToolOptions:= TToolOptions.Create;
end;

//------------------------------------------------------------------------------

procedure TPictureEngine.AfterConstruction;
begin
  // only here does Self become a valid reference -> can register other parts
  FUndoEngine:= TUndoEngine.Create(Self);
  FBackgroundEngine:= TBackgroundEngine.Create;
  FBackgroundEngine.OnChange:= Self.BackgroundChangeHandler;
end;

//------------------------------------------------------------------------------

destructor TPictureEngine.Destroy;
begin
  FPctOriginal.Free;
  FPctOverlay.Free;
  FPctBuffer.Free;
  FToolOptions.Free;
  FBackgroundEngine.Free;
  FUndoEngine.Free;
  inherited Destroy;
end;

//------------------------------------------------------------------------------

procedure TPictureEngine.Reloaded;
begin
  // refresh all, got 100% new picture
  MakeTransparent(FPctOriginal, BackClrOpaque, FPctOriginal.BoundsRect);

  FPctBuffer.SetSizeFrom(FPctOriginal);
  FPctOverlay.SetSizeFrom(FPctOriginal);

  FBackgroundEngine.Size:= FPctOriginal.BoundsRect; // no callback issued

  ResyncOverlay;

  RepaintBuffer(BoundsRect);
  Paint;

  FUndoEngine.Restart;
end;

//------------------------------------------------------------------------------

procedure TPictureEngine.SetViewport(ANew: TImgView32);
begin
  FViewport:= ANew;
  FViewport.Bitmap.SetSizeFrom(FPctBuffer);
  Paint;
  // repaint tool?
end;

//------------------------------------------------------------------------------

procedure TPictureEngine.SetViewMode(AMode: TViewMode);
begin
  if AMode <> FViewMode then begin // only react if different (avoiding big redraw)
    FViewMode:= AMode; // set internal
    RepaintBuffer(BoundsRect);
    Paint;
  end;
end;

//------------------------------------------------------------------------------

procedure TPictureEngine.SetOptions(const AOptions: TSdiEdOptions);
var Diff: TOptionsCompare;
    OldCallback: Boolean;
begin
  Diff:= DiffOptions(FOptions, AOptions); // change between old and new states
  FOptions:= AOptions; // the next two Repaint* calls take values from FOptions,
  // so it must be already set here, or they redraw with old values and it seems
  // as if nothing happened.
  OldCallback:= FDoCallBack; FDoCallBack:= False; // backup and disable
  FBackgroundEngine.Options:= FOptions;
  // first propagate to picture maker (while disabled callback)...
  if Diff.HighlightChanged or Diff.NightChanged then begin
    RepaintBuffer(BoundsRect);
    Paint;
  end;
  // ...then redraw self with parts from it that already have new look, flush to
  // viewport and while at this, save one complete redraw by having switched off
  // callback - that's why it must be switched off!
  FDoCallBack:= OldCallback; // finally return callback to what it was
  if Diff.UndoChanged then with FUndoEngine, FOptions.UndoOptions do begin
    MaxCount:= MaxLevel;
    RestrictCount:= Restrict;
  end; // undo engine is not so important as it does not concern drawing
end;

//------------------------------------------------------------------------------

procedure TPictureEngine.Paint;
begin
  if Assigned(FViewport) then begin
    FViewport.Bitmap.SetSizeFrom(FPctOriginal);
    RedrawRect(BoundsRect);
  end;
end;

//------------------------------------------------------------------------------
// updates FPctBuffer with the proper item to show

procedure TPictureEngine.RepaintBuffer(const APlace: TRect);
begin
  FPctOriginal.DrawMode:= dmOpaque;
  // now we want to keep original showing through though!
  FPctBuffer.Draw(APlace, APlace, FPctOriginal);
  // now flush also proper image

  FPctOverlay.DrawMode:= dmBlend;
  FPctBuffer.Draw(APlace, APlace, FPctOverlay);

  MakeTransparent(FPctBuffer, BackClrOpaque, APlace);

  case FViewMode of
    vmDark:      MakeDarkened(FPctBuffer, FOptions, APlace);
    vmHighlight: MakeHighlighted(FPctBuffer, FOptions, APlace);
    else         { nothing };
  end;
end;

//------------------------------------------------------------------------------

procedure TPictureEngine.BackgroundChangeHandler(Sender: TObject);
begin
  if Sender = FBackgroundEngine then begin // only if it's our own good old servant
    if FDoCallBack then begin // and then still only if allowed
      RepaintBuffer(BoundsRect);
      Paint;
    end;
  end else raise Exception.CreateFmt(
    'Foreign call to TPictureEngine.BackgroundChangeHandler from %s',
    [Sender.ClassName]);
end;

//------------------------------------------------------------------------------

procedure TPictureEngine.RedrawRect(const APlace: TRect);
var R: TRect;
begin
  if IntersectRect(R, APlace, FPctOriginal.BoundsRect) then begin
    // do it only for part inside picture, and only if it exists at all
    FBackgroundEngine.Render(Viewport.Bitmap, FViewMode, dmOpaque, R);

    FPctBuffer.DrawMode:= dmBlend;
    Viewport.Bitmap.Draw(R, R, FPctBuffer);
  end;
end;

//------------------------------------------------------------------------------

procedure TPictureEngine.UpdateRect(const APlace: TRect);
var R: TRect;
begin
  if IntersectRect(R, APlace, FPctOriginal.BoundsRect) then begin
    // same safeguard as in RedrawRect
    RepaintBuffer(R);
  end;
end;

//------------------------------------------------------------------------------

procedure TPictureEngine.CommitOverlay(const AWhere: array of TRect);
var AllPlaces: TRect;
    i: Integer;
begin
  AllPlaces:= ZeroRect;
  for i:= Low(AWhere) to High(AWhere) do
    AllPlaces:= AccumulateRect(AllPlaces, AWhere[i]);
  FUndoEngine.AddStep(AllPlaces);
  FPctOriginal.DrawMode:= dmBlend;
  for i:= Low(AWhere) to High(AWhere) do begin
    FPctOverlay.DrawMode:= dmBlend;
    FPctOriginal.Draw(AWhere[i], AWhere[i], FPctOverlay);
    FPctOverlay.DrawMode:= dmOpaque;
    with AWhere[i] do FPctOverlay.FillRect(Left, Top, Right, Bottom, BackClrTransparent);
    UpdateRect(AWhere[i]);
  end;
  UpdateRect(FPctOriginal.BoundsRect);
  MakeTransparent(FPctOriginal, BackClrOpaque, FPctOriginal.BoundsRect);
  if Assigned(FOnChange) then FOnChange(Self);
end;

//------------------------------------------------------------------------------

procedure TPictureEngine.ResyncOverlay;
// after size changed or such
begin
  with FPctOverlay do begin
    OuterColor:= BackClrTransparent;
    SetSizeFrom(FPctOriginal);
    // remember: setsize only allocates memory
    Clear(BackClrTransparent);
  end;
end;

//------------------------------------------------------------------------------

procedure TPictureEngine.DiscardOverlay;
begin
  FPctOverlay.Clear(BackClrTransparent);
end;

//------------------------------------------------------------------------------

function TPictureEngine.GetBoundsRect: TRect;
begin
  Result:= FPctOriginal.BoundsRect;
end;

//------------------------------------------------------------------------------

procedure TPictureEngine.CopyOverlay;
begin
  FPctOriginal.DrawMode:= dmOpaque;
  FPctOriginal.DrawTo(FPctOverlay, 0, 0);
end;

//------------------------------------------------------------------------------

procedure TPictureEngine.Dump;
begin
  FPctOriginal.SaveToFile('0_FPctOriginal.bmp');
  FPctBuffer.SaveToFile('3_FPctBuffer.bmp');
  FPctOverlay.SaveToFile('1_FPctOverlay.bmp');
end;

//------------------------------------------------------------------------------

end.
