(*************************************************************************
 *  BackPicU.pas + BackPicU.dfm                                          *
 *  Vladimr Slvik 2005-10                                              *
 *  Delphi 7 Personal                                                    *
 *  cp1250                                                               *
 *                                                                       *
 *  background picture manager form of Shades :                          *
 *    list of pics in application library, preview                       *
 *                                                                       *
 *  -additional libraries: Graphics32, PNGImage                          *
 *************************************************************************)

unit BackPicU;

{$INCLUDE Switches.inc}
{t default -}

interface

uses
  Windows, SysUtils, Controls, Forms, StdCtrls, ExtCtrls, Classes,
  GR32, GR32_Image, BackgroundEngineU;

type
  TBackPicFrm = class(TForm)
    gPrev: TImage32;
    btnAdd: TButton;
    rbtnPicture: TRadioButton;
    rbtnColor: TRadioButton;
    edtColor: TEdit;
    btnColor: TButton;
    shpColor: TShape;
    btnOk: TButton;
    btnCancel: TButton;
    btnDefaultColor: TButton;
    lbPictures: TListBox;
    btnDel: TButton;
    Bevel1: TBevel;
    btnRename: TButton;
    procedure btnAddClick(Sender: TObject);
    procedure btnColorClick(Sender: TObject);
    procedure btnDefaultColorClick(Sender: TObject);
    procedure btnDelClick(Sender: TObject);
    procedure lbPicturesClick(Sender: TObject);
    procedure btnOkClick(Sender: TObject);
    procedure FormCreate(Sender: TObject);
    procedure btnRenameClick(Sender: TObject);
    procedure FormKeyDown(Sender: TObject; var Key: Word;
      Shift: TShiftState);
    procedure lbPicturesDblClick(Sender: TObject);
  private
    FOpenedFile: String;
    function GetBackPicture: TBitmap32;
    procedure OpenFile(const FileName: String);
    function GetBackColor: TColor32;
    procedure SetBackColor(const AColor: TColor32);
    function GetBkSetting: TBackgroundSetting;
    procedure SetBkSetting(const ASetting: TBackgroundSetting);
  public
    property BackPicture: TBitmap32 read GetBackPicture;
    property BackColor: TColor32 read GetBackColor write SetBackColor;
    property SelectedState: TBackgroundSetting read GetBkSetting write SetBkSetting;
    procedure ReloadFileList;
  end;

var
  BackPicFrm: TBackPicFrm;

//==============================================================================

implementation

{$R *.dfm}

uses Math, ShellAPI, Dialogs, // Delphi
     PNGImage, // libs
     ColorU, StrU, TranslationU, TransGuiU, // units
     ModuleU, MainU, ConstStrU, OptionU, ConfigU, CoreLowU, ShadesInOutU, SysLowU; // project

//------------------------------------------------------------------------------

function TBackPicFrm.GetBackPicture: TBitmap32;
begin
  Result:= gPrev.Bitmap;
end;

//------------------------------------------------------------------------------

procedure TBackPicFrm.OpenFile(const FileName: String);
var P: TPNGObject;
begin
  P:= TPNGObject.Create;
  try
    P.LoadFromFile(GoodSlashes(FileName));
    gPrev.Bitmap.Assign(P);
    FOpenedFile:= FileName;
  finally
    P.Free;
  end;
end;

//------------------------------------------------------------------------------

procedure TBackPicFrm.btnAddClick(Sender: TObject);
// Add file = copy to safe data store with randomized name to prevent duplicate
// problems (deleting etc.), add to list, highlight in listbox and show preview.
var ShortName, NewShortName, OldFile, NewFile, NewDir: String;
    NewPos: Integer;
begin
  with DialogModule.OpenPictureDialog, lbPictures do if Execute then begin
    OldFile:= FileName;
    if FileIsPNG(OldFile) then begin
      ShortName:= ExtractFilename(OldFile);
      NewShortName:= CleanString(ShortName, '=', '_');
      // INI files think first = is separator
      case MainFrm.Options.IniFileKind of
        iftFile: begin
          repeat
            NewFile:= GoodSlashes(DataPath + '/Backgrounds/'
              + IntToHex(RandomRange(Low(Integer), High(Integer)), 8)
              + '_'+ NewShortName);
          until not FileExists(NewFile); // find some random file name
          NewDir:= ExtractFileDir(NewFile);
          if not DirectoryExists(NewDir) then CreateDir(NewDir);
          CopyFile(PChar(OldFile), PChar(NewFile), False);
        end;
        iftRegistry, iftNone: begin
          NewFile:= OldFile;
          if Pos('=', NewFile) <> 0 then begin
            Application.MessageBox(PChar(_('Files with = in their path cannot be used in Windows Registry data mode.')), //t+
              PChar(sscMsgCapError), MB_OK + MB_ICONERROR);
            Abort;
          end;
        end;
      end;
      NewPos:= MainFrm.AddBackFile(NewFile, ShortName);
      ReloadFileList;
      ItemIndex:= NewPos;
      OpenFile(MainFrm.BackFiles.ValueFromIndex[NewPos]);
    end else Application.MessageBox(
      PChar(_('The file is not a valid PNG picture.')), //t+
      PChar(Translate(sscMsgCapError)), MB_OK + MB_ICONERROR);
  end;
end;

//------------------------------------------------------------------------------

procedure TBackPicFrm.btnColorClick(Sender: TObject);
begin
  with DialogModule.ColorDialog do begin
    Color:= shpColor.Brush.Color;
    if Execute then begin
      shpColor.Brush.Color:= Color;
      edtColor.Text:= Cl32ToHex(Color32(Color));
      rbtnColor.Checked:= True;
    end;
  end;
end;

//------------------------------------------------------------------------------

procedure TBackPicFrm.btnDefaultColorClick(Sender: TObject);
begin
 edtColor.Text:= 'ffe7ffff';
 shpColor.Brush.Color:= WinColor($FFE7FFFF);
 rbtnColor.Checked:= True;
end;

//------------------------------------------------------------------------------

procedure TBackPicFrm.btnDelClick(Sender: TObject);
var OldIndex, R: Integer;
    FileName: String;
  //............................................................................
  procedure KillFile;
  begin
    if Pos(GoodSlashes(DataPath + '/Backgrounds/'), FileName) <> 0 then begin
      // is in our own repository, delete without remorse
      DeleteFile(FileName)
    end else begin
      // is outside, ask what to do
      R:= Application.MessageBox(
        PChar(_('The background will be removed from list, but its file is not in data repository. It is linked from somewhere outside and may be used by other programs. Do you want to keep it?') //t+
          + #10#10 + _('Note: if you choose not to keep it, you can always return it from recycle bin.')), //t+
        PChar(Translate(sscMsgCapProceed)), MB_ICONWARNING + MB_YESNO);
      if R = ID_NO then DeleteFileSystemObject(Handle, FileName, False);
      // delete to recycle bin so that user can later undo if needed
    end;
  end;
  //............................................................................
begin
  with lbPictures do if ItemIndex > -1 then begin
    OldIndex:= ItemIndex; // keep backup
    FileName:= GoodSlashes(MainFrm.BackFiles.Names[OldIndex]);
    if MainFrm.Options.IniFileKind = iftFile then begin
      if MainFrm.Options.InterfaceOptions.ConfirmBkDelete then begin
        R:= Application.MessageBox('Do you really want to delete this background picture?', //t+
          PChar(Translate(sscMsgCapProceed)), MB_YESNO + MB_ICONWARNING);
        if ID_YES = R then KillFile;
      end else KillFile;
    end;
    MainFrm.BackFiles.Delete(OldIndex);
    ReloadFileList;
    ItemIndex:= EnsureRange(OldIndex, -1, Count - 1);
    // make sure ItemIndex is within range; can be -1, too
    if ItemIndex > -1 then begin // if there are some pictures left in list
      FileName:= MainFrm.BackFiles.Names[ItemIndex];
      if FOpenedFile <> FileName then OpenFile(FileName);
    end else gPrev.Bitmap.SetSize(0, 0); // if not, clean
  end;
end;

//------------------------------------------------------------------------------

procedure TBackPicFrm.lbPicturesClick(Sender: TObject);
begin
  with lbPictures do if ItemIndex > -1 then begin
    OpenFile(MainFrm.BackFiles.Names[ItemIndex]);
    rbtnPicture.Checked:= True;
  end else
    gPrev.Bitmap.SetSize(0, 0);
end;

//------------------------------------------------------------------------------

procedure TBackPicFrm.btnOkClick(Sender: TObject);
begin
  if rbtnPicture.Checked and gPrev.Bitmap.Empty then
    ShowMessage('No picture selected!') else ModalResult:= mrOk; //t+
end;

//------------------------------------------------------------------------------

procedure TBackPicFrm.FormCreate(Sender: TObject);
begin
  lbPictures.Items.NameValueSeparator:= '|';
  TranslateComponent(Self);
  Font.Name:= Translate(sscFontName);
end;

//------------------------------------------------------------------------------

procedure TBackPicFrm.ReloadFileList;
var i: Integer;
begin
 lbPictures.Clear;
  with MainFrm.BackFiles do if Count > 0 then for i:= 0 to Count - 1 do
    lbPictures.Items.Add(ValueFromIndex[i]);
end;

//------------------------------------------------------------------------------

procedure TBackPicFrm.btnRenameClick(Sender: TObject);
var BackName, BackFile: String;
    ItemIndex: Integer;
begin
  ItemIndex:= lbPictures.ItemIndex; // cache
  if ItemIndex > -1 then with MainFrm.BackFiles do begin
    BackName:= ValueFromIndex[ItemIndex];
    BackFile:= Names[ItemIndex];
    if InputQuery(_('Rename background'), //t+
        _('Please choose new name:'), BackName) then begin //t+
      ValueFromIndex[ItemIndex]:= BackName;
      MainFrm.SortBackFilesByDisplayName;
      ItemIndex:= IndexOfName(BackFile);
      ReloadFileList;
      lbPictures.ItemIndex:= ItemIndex;
    end;
  end;
end;

//------------------------------------------------------------------------------

procedure TBackPicFrm.FormKeyDown(Sender: TObject; var Key: Word;
  Shift: TShiftState);
begin
  if Key = VK_Return then begin
    if ActiveControl = edtColor then begin
      shpColor.Brush.Color:= WinColor(HexToCl32(edtColor.Text));
      rbtnColor.Checked:= True;
    end else
      btnOkClick(Sender);
  end;
end;

//------------------------------------------------------------------------------

procedure TBackPicFrm.lbPicturesDblClick(Sender: TObject);
begin
  with lbPictures do if ItemIndex > -1 then begin
    OpenFile(MainFrm.BackFiles.Names[ItemIndex]);
    rbtnPicture.Checked:= True;
    ModalResult:= mrOk;
  end else
    gPrev.Bitmap.SetSize(0, 0);
end;

//------------------------------------------------------------------------------

function TBackPicFrm.GetBackColor: TColor32;
begin
  Result:= Color32(shpColor.Brush.Color);
end;

procedure TBackPicFrm.SetBackColor(const AColor: TColor32);
begin
  shpColor.Brush.Color:= WinColor(AColor);
end;

//------------------------------------------------------------------------------

function TBackPicFrm.GetBkSetting: TBackgroundSetting;
begin
  if rbtnColor.Checked then Result:= bsColor else Result:= bsPicture;
end;

procedure TBackPicFrm.SetBkSetting(const ASetting: TBackgroundSetting);
begin
  case ASetting of
    bsColor: rbtnColor.Checked:= True;
    bsPicture: rbtnPicture.Checked:= True;
  end;
end;

//------------------------------------------------------------------------------

end.
